<?php

namespace MediaLibrary\Services;

use MediaLibrary\Models\Media;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;

class MediaService
{
    /**
     * Upload a file and create a Media record
     */
    public function uploadFile(UploadedFile $file, ?int $staffId = null, array $metadata = []): Media
    {
        $year = date('Y');
        $month = date('n'); // 1-12 format
        
        // Create WordPress-style directory structure
        $directory = "media/{$year}/" . str_pad($month, 2, '0', STR_PAD_LEFT);
        
        // Generate unique filename while preserving extension
        $extension = $file->getClientOriginalExtension();
        $filename = $this->generateUniqueFilename($file->getClientOriginalName(), $directory, $extension);
        
        // Store the file
        $filePath = $file->storeAs($directory, $filename, 'public');
        
        // Create Media record
        $media = Media::create([
            'file_name' => $filename,
            'file_path' => $filePath,
            'mime_type' => $file->getMimeType(),
            'file_size' => $file->getSize(),
            'title' => $metadata['title'] ?? null,
            'alt_text' => $metadata['alt_text'] ?? null,
            'caption' => $metadata['caption'] ?? null,
            'description' => $metadata['description'] ?? null,
            'link_url' => $metadata['link_url'] ?? null,
            'link_target' => $metadata['link_target'] ?? '_self',
            'staff_id' => $staffId,
            'disk' => 'public',
            'year_month_path' => date('Y/m'),
        ]);

        // Generate thumbnails for images
        if ($media->isImage()) {
            $this->generateThumbnails($media);
        }

        return $media;
    }

    /**
     * Generate a unique filename to avoid conflicts
     */
    protected function generateUniqueFilename(string $originalName, string $directory, string $extension): string
    {
        $baseName = pathinfo($originalName, PATHINFO_FILENAME);
        $sanitizedBaseName = Str::slug($baseName);
        
        // If sanitized name is empty, use timestamp
        if (empty($sanitizedBaseName)) {
            $sanitizedBaseName = 'file-' . time();
        }
        
        $filename = $sanitizedBaseName . '.' . $extension;
        $fullPath = $directory . '/' . $filename;
        
        // Check if file exists and generate unique name
        $counter = 1;
        while (Storage::disk('public')->exists($fullPath)) {
            $filename = $sanitizedBaseName . '-' . $counter . '.' . $extension;
            $fullPath = $directory . '/' . $filename;
            $counter++;
        }
        
        return $filename;
    }

    /**
     * Generate thumbnails for images
     */
    protected function generateThumbnails(Media $media): void
    {
        if (!$media->isImage()) {
            return;
        }

        $fullPath = $media->full_path;
        $directory = dirname($media->file_path);
        $filename = pathinfo($media->file_name, PATHINFO_FILENAME);
        $extension = $media->extension;

        // Generate different thumbnail sizes
        $thumbnailSizes = [
            'thumbnail' => [150, 150],
            'medium' => [300, 300],
            'large' => [1024, 1024],
        ];

        foreach ($thumbnailSizes as $sizeName => $dimensions) {
            $thumbnailPath = $directory . '/' . $filename . '-' . $sizeName . '.' . $extension;
            
            try {
                $image = Image::make($fullPath);
                $image->resize($dimensions[0], $dimensions[1], function ($constraint) {
                    $constraint->aspectRatio();
                    $constraint->upsize();
                });
                
                $image->save(Storage::disk('public')->path($thumbnailPath));
            } catch (\Exception $e) {
                \Log::error('Failed to generate thumbnail', [
                    'media_id' => $media->id,
                    'size' => $sizeName,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * Delete a media file and its thumbnails
     */
    public function deleteMedia(Media $media): bool
    {
        try {
            // Delete main file
            Storage::disk('public')->delete($media->file_path);
            
            // Delete thumbnails if it's an image
            if ($media->isImage()) {
                $directory = dirname($media->file_path);
                $filename = pathinfo($media->file_name, PATHINFO_FILENAME);
                $extension = $media->extension;
                
                $thumbnailSizes = ['thumbnail', 'medium', 'large'];
                foreach ($thumbnailSizes as $sizeName) {
                    $thumbnailPath = $directory . '/' . $filename . '-' . $sizeName . '.' . $extension;
                    Storage::disk('public')->delete($thumbnailPath);
                }
            }
            
            // Delete the database record
            $media->delete();
            
            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to delete media', [
                'media_id' => $media->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Get thumbnail URL for a media item
     */
    public function getThumbnailUrl(Media $media, string $size = 'thumbnail'): ?string
    {
        if (!$media->isImage()) {
            return null;
        }

        $directory = dirname($media->file_path);
        $filename = pathinfo($media->file_name, PATHINFO_FILENAME);
        $extension = $media->extension;
        $thumbnailPath = $directory . '/' . $filename . '-' . $size . '.' . $extension;

        if (Storage::disk('public')->exists($thumbnailPath)) {
            return Storage::disk('public')->url($thumbnailPath);
        }

        // Fallback to original image if thumbnail doesn't exist
        return $media->url;
    }

    /**
     * Update media metadata
     */
    public function updateMetadata(Media $media, array $metadata): Media
    {
        $media->update([
            'title' => $metadata['title'] ?? $media->title,
            'alt_text' => $metadata['alt_text'] ?? $media->alt_text,
            'caption' => $metadata['caption'] ?? $media->caption,
            'description' => $metadata['description'] ?? $media->description,
            'link_url' => $metadata['link_url'] ?? $media->link_url,
            'link_target' => $metadata['link_target'] ?? $media->link_target,
        ]);

        return $media;
    }

    /**
     * Get media statistics
     */
    public function getMediaStats(): array
    {
        return [
            'total_files' => Media::count(),
            'total_size' => Media::sum('file_size'),
            'images_count' => Media::images()->count(),
            'videos_count' => Media::videos()->count(),
            'documents_count' => Media::documents()->count(),
        ];
    }

    /**
     * Clean up orphaned files (files in storage but not in database)
     */
    public function cleanupOrphanedFiles(): int
    {
        $cleanedCount = 0;
        $mediaDirectory = 'media';
        
        if (!Storage::disk('public')->exists($mediaDirectory)) {
            return 0;
        }

        $files = Storage::disk('public')->allFiles($mediaDirectory);
        
        foreach ($files as $file) {
            // Skip thumbnails (they will be cleaned up with their parent)
            if (str_contains($file, '-thumbnail.') || 
                str_contains($file, '-medium.') || 
                str_contains($file, '-large.')) {
                continue;
            }

            // Check if this file exists in database
            $exists = Media::where('file_path', $file)->exists();
            
            if (!$exists) {
                // Delete the file and its thumbnails
                Storage::disk('public')->delete($file);
                
                // Delete potential thumbnails
                $pathInfo = pathinfo($file);
                $thumbnailSizes = ['thumbnail', 'medium', 'large'];
                foreach ($thumbnailSizes as $size) {
                    $thumbnailPath = $pathInfo['dirname'] . '/' . $pathInfo['filename'] . '-' . $size . '.' . $pathInfo['extension'];
                    Storage::disk('public')->delete($thumbnailPath);
                }
                
                $cleanedCount++;
            }
        }

        return $cleanedCount;
    }
}

