<?php

namespace MediaLibrary\Resources\MediaResource\Pages;

use MediaLibrary\Resources\MediaResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use MediaLibrary\Services\MediaService;
use Filament\Notifications\Notification;

class EditMedia extends EditRecord
{
    protected static string $resource = MediaResource::class;

    public function isImage(): bool
    {
        if (!$this->record) {
            return false;
        }
        
        return $this->record->isImage();
    }

    protected function getHeaderActions(): array
    {
        return [
            Actions\DeleteAction::make()
                ->action(function () {
                    $mediaService = app(MediaService::class);
                    $mediaService->deleteMedia($this->record);
                    
                    Notification::make()
                        ->title('Media deleted')
                        ->body("File '" . ($this->record->title ?: $this->record->file_name) . "' has been deleted.")
                        ->success()
                        ->send();
                    
                    $this->redirect($this->getResource()::getUrl('index'));
                }),
        ];
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        // Remove any file upload data as we're just editing metadata
        unset($data['file']);
        
        return $data;
    }

    protected function handleRecordUpdate($record, array $data): \Illuminate\Database\Eloquent\Model
    {
        $mediaService = app(MediaService::class);
        
        $metadata = [
            'title' => $data['title'] ?? $record->title,
            'alt_text' => $data['alt_text'] ?? $record->alt_text,
            'description' => $data['description'] ?? $record->description,
        ];
        
        $mediaService->updateMetadata($record, $metadata);
        
        Notification::make()
            ->title('Media updated successfully')
            ->body("Metadata for '" . ($record->title ?: $record->file_name) . "' has been updated.")
            ->success()
            ->send();
            
        return $record;
    }
}

