<?php

namespace MediaLibrary\Resources\MediaResource\Pages;

use MediaLibrary\Resources\MediaResource;
use MediaLibrary\Models\Media;
use MediaLibrary\Services\MediaService;
use Filament\Resources\Pages\CreateRecord;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Storage;
use Filament\Forms\Components\FileUpload;

class CreateMedia extends CreateRecord
{
    protected static string $resource = MediaResource::class;

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // Remove the file upload from data as we'll handle it separately
        unset($data['file']);

        return $data;
    }

    protected function handleRecordCreation(array $data): Media
    {
        $fileUpload = $this->data['file'] ?? null;
        
        if (!$fileUpload) {
            throw new \Exception('No file uploaded');
        }

        // Get the current staff member ID
        $staffId = auth()->user()->staff->id ?? null;
        
        // Get MediaService
        $mediaService = app(MediaService::class);
        
        // Handle file upload
        $uploadedFile = null;
        
        if (is_array($fileUpload)) {
            if (empty($fileUpload)) {
                throw new \Exception('File upload array is empty');
            }
            // Get the first file path from the array
            $fileKey = array_keys($fileUpload)[0];
            $filePath = $fileUpload[$fileKey];
            
            // Get the file from storage
            $tempPath = Storage::disk('public')->path($filePath);
            
            // Create UploadedFile instance from temp file
            $uploadedFile = new \Illuminate\Http\UploadedFile(
                $tempPath,
                basename($filePath),
                mime_content_type($tempPath),
                null,
                true // test mode
            );
        } else {
            $uploadedFile = $fileUpload;
        }

        // Prepare metadata
        $metadata = [
            'title' => $data['title'] ?? null,
            'alt_text' => $data['alt_text'] ?? null,
            'description' => $data['description'] ?? null,
        ];

        // Upload file using MediaService
        $media = $mediaService->uploadFile($uploadedFile, $staffId, $metadata);

        Notification::make()
            ->title('Media uploaded successfully')
            ->body("File '{$media->file_name}' has been uploaded and organized.")
            ->success()
            ->send();

        return $media;
    }
}

