<?php

namespace MediaLibrary\Resources;

use MediaLibrary\Resources\MediaResource\Pages;
use MediaLibrary\Models\Media;
use MediaLibrary\Services\MediaService;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables\Table;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Illuminate\Support\Facades\Storage;
use Filament\Forms\Components\Textarea;
use Filament\Tables\Columns\ImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Actions\BulkAction;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\Filter;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;

class MediaResource extends Resource
{
    protected static ?string $model = Media::class;

    protected static ?string $navigationIcon = 'heroicon-o-photo';
    
    protected static ?string $navigationGroup = 'Content';
    
    protected static ?string $navigationLabel = 'Media';
    
    protected static ?int $navigationSort = 30;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('File Upload')
                    ->description('Upload a new media file')
                    ->schema([
                        FileUpload::make('file')
                            ->label('Choose File')
                            ->acceptedFileTypes(['image/*', 'video/*', 'application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', 'text/plain'])
                            ->maxSize(10240) // 10MB
                            ->directory('temp-uploads')
                            ->visibility('public')
                            ->disk('public')
                            ->required()
                            ->preserveFilenames()
                            ->live()
                            ->afterStateUpdated(function ($state, $set, $get, $record) {
                                if ($state && !$record) {
                                    // Auto-populate alt text from filename for new uploads
                                    if (is_array($state)) {
                                        $firstKey = array_keys($state)[0] ?? null;
                                        if ($firstKey) {
                                            $filename = pathinfo($state[$firstKey], PATHINFO_FILENAME);
                                            $set('alt_text', ucwords(str_replace(['-', '_'], ' ', $filename)));
                                        }
                                    }
                                }
                            }),
                    ])
                    ->visible(fn ($record) => !$record), // Only show for new uploads

                Section::make('Media Details')
                    ->description('Information about the media file')
                    ->schema([
                        TextInput::make('title')
                            ->label('Title')
                            ->placeholder('Enter a title for this media')
                            ->helperText('A descriptive title for the media')
                            ->maxLength(255),
                        
                        TextInput::make('alt_text')
                            ->label('Alt Text')
                            ->placeholder('Describe the image for accessibility')
                            ->helperText('Important for screen readers and SEO')
                            ->maxLength(255),
                        
                        Textarea::make('description')
                            ->label('Description')
                            ->placeholder('Optional description of the media')
                            ->rows(3)
                            ->maxLength(1000),
                        
                        TextInput::make('link_url')
                            ->label('Media URL')
                            ->disabled()
                            ->dehydrated(false)
                            ->formatStateUsing(function ($record) {
                                if (!$record) return 'Will be generated after upload';
                                return $record ? url('storage/' . $record->file_path) : 'Not available';
                            })
                            ->columnSpanFull()
                            ->extraInputAttributes([
                                'style' => 'font-family: monospace; font-size: 0.875rem;'
                            ]),
                    ])
                    ->columns(2)
                    ->visible(fn ($record) => $record), // Only show for existing records
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                ImageColumn::make('preview')
                    ->label('Preview')
                    ->getStateUsing(function (Media $record) {
                        if ($record->isImage()) {
                            return $record->url;
                        }
                        return null; // Will show default file icon
                    })
                    ->size(120)
                    ->circular(false)
                    ->url(null),

                TextColumn::make('title')
                    ->label('Title')
                    ->searchable()
                    ->sortable()
                    ->limit(30)
                    ->placeholder('Untitled')
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 30 ? $state : null;
                    }),

                TextColumn::make('human_file_size')
                    ->label('Size')
                    ->getStateUsing(fn (Media $record) => $record->human_file_size)
                    ->sortable(query: function (Builder $query, string $direction): Builder {
                        return $query->orderBy('file_size', $direction);
                    }),

                TextColumn::make('upload_date')
                    ->label('Uploaded')
                    ->getStateUsing(function (Media $record) {
                        return $record->created_at->format('M j, Y');
                    })
                    ->sortable(query: function (Builder $query, string $direction): Builder {
                        return $query->orderBy('created_at', $direction);
                    }),

                TextColumn::make('alt_text')
                    ->label('Alt Text')
                    ->limit(20)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();
                        return strlen($state) > 20 ? $state : null;
                    })
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('staff.firstname')
                    ->label('Uploaded By')
                    ->formatStateUsing(function ($record) {
                        if ($record->staff) {
                            return $record->staff->firstname . ' ' . $record->staff->lastname;
                        }
                        return 'Unknown';
                    })
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('file_type')
                    ->label('File Type')
                    ->options([
                        'image' => 'Images',
                        'video' => 'Videos',
                        'document' => 'Documents',
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return match ($data['value'] ?? null) {
                            'image' => $query->images(),
                            'video' => $query->videos(),
                            'document' => $query->documents(),
                            default => $query,
                        };
                    }),

                SelectFilter::make('year')
                    ->label('Year')
                    ->options(function () {
                        $years = Media::getAvailableYears()->toArray();
                        return array_combine($years, $years);
                    }),

                SelectFilter::make('month')
                    ->label('Month')
                    ->options(function () {
                        return Media::getAvailableMonths()->toArray();
                    }),

                Filter::make('upload_date')
                    ->label('Upload Date')
                    ->form([
                        DatePicker::make('uploaded_from')
                            ->label('From'),
                        DatePicker::make('uploaded_until')
                            ->label('Until'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['uploaded_from'],
                                fn (Builder $query, $date): Builder => $query->whereDate('created_at', '>=', $date),
                            )
                            ->when(
                                $data['uploaded_until'],
                                fn (Builder $query, $date): Builder => $query->whereDate('created_at', '<=', $date),
                            );
                    }),
            ])
            ->actions([
                Action::make('edit')
                    ->label('Edit')
                    ->icon('heroicon-o-pencil')
                    ->url(fn (Media $record): string => MediaResource::getUrl('edit', ['record' => $record])),

                DeleteAction::make()
                    ->action(function (Media $record) {
                        $mediaService = app(MediaService::class);
                        $mediaService->deleteMedia($record);
                    }),
            ])
            ->bulkActions([
                BulkAction::make('delete')
                    ->label('Delete Selected')
                    ->icon('heroicon-o-trash')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->action(function (Collection $records) {
                        $mediaService = app(MediaService::class);
                        foreach ($records as $record) {
                            $mediaService->deleteMedia($record);
                        }
                    }),
            ])
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading('No media files')
            ->emptyStateDescription('Upload your first media file to get started.')
            ->emptyStateIcon('heroicon-o-photo');
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListMedia::route('/'),
            'create' => Pages\CreateMedia::route('/create'),
            'edit' => Pages\EditMedia::route('/{record}/edit'),
        ];
    }
}

