<?php

namespace MediaLibrary\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Storage;

/**
 * Media Model
 * 
 * Represents uploaded media files with WordPress-style organization
 * and staff attribution. Uses our own custom_media table.
 */
class Media extends Model
{
    use HasFactory;

    /**
     * The table associated with the model.
     */
    protected $table = 'custom_media';

    /**
     * The attributes that are mass assignable.
     */
    protected $fillable = [
        'title',
        'alt_text',
        'caption',
        'description',
        'link_url',
        'link_target',
        'staff_id',
        'file_name',
        'file_path',
        'mime_type',
        'file_size',
        'disk',
        'year_month_path',
    ];

    /**
     * The attributes that should be cast.
     */
    protected $casts = [
        'staff_id' => 'integer',
        'file_size' => 'integer',
    ];

    /**
     * Boot the model and register model events
     */
    protected static function boot()
    {
        parent::boot();

        // Auto-set staff_id when creating
        static::creating(function ($media) {
            if (!$media->staff_id && auth()->check()) {
                // Try to get staff ID from authenticated user
                $user = auth()->user();
                if ($user && method_exists($user, 'staff')) {
                    $media->staff_id = $user->staff?->id;
                }
            }
            
            // Auto-set year_month_path for WordPress-style organization
            if (!$media->year_month_path) {
                $media->year_month_path = date('Y/m');
            }
        });
    }

    /**
     * Relationship to Staff
     */
    public function staff()
    {
        return $this->belongsTo(\Lunar\Admin\Models\Staff::class, 'staff_id');
    }

    /**
     * Get the WordPress-style path attribute
     */
    public function getWordPressPathAttribute()
    {
        return $this->year_month_path . '/' . $this->file_name;
    }

    /**
     * Get the full URL to the media file
     */
    public function getUrlAttribute()
    {
        return Storage::disk($this->disk)->url($this->file_path);
    }

    /**
     * Get the full path to the media file
     */
    public function getFullPathAttribute()
    {
        return Storage::disk($this->disk)->path($this->file_path);
    }

    /**
     * Get the file extension
     */
    public function getExtensionAttribute()
    {
        return pathinfo($this->file_name, PATHINFO_EXTENSION);
    }

    /**
     * Check if the media is an image
     */
    public function isImage()
    {
        return str_starts_with($this->mime_type, 'image/');
    }

    /**
     * Check if the media is a video
     */
    public function isVideo()
    {
        return str_starts_with($this->mime_type, 'video/');
    }

    /**
     * Check if the media is a document
     */
    public function isDocument()
    {
        return in_array($this->mime_type, [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * Get human readable file size
     */
    public function getHumanFileSizeAttribute()
    {
        if (!$this->file_size) {
            return 'Unknown';
        }

        $bytes = $this->file_size;
        $units = ['B', 'KB', 'MB', 'GB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, 2) . ' ' . $units[$i];
    }

    /**
     * Get original filename (alias for file_name)
     */
    public function getOriginalFilenameAttribute()
    {
        return $this->file_name;
    }

    /**
     * Scope for images only
     */
    public function scopeImages(Builder $query): Builder
    {
        return $query->where('mime_type', 'like', 'image/%');
    }

    /**
     * Scope for videos only
     */
    public function scopeVideos(Builder $query): Builder
    {
        return $query->where('mime_type', 'like', 'video/%');
    }

    /**
     * Scope for documents only
     */
    public function scopeDocuments(Builder $query): Builder
    {
        return $query->whereIn('mime_type', [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    /**
     * Scope for date range filtering
     */
    public function scopeByDateRange(Builder $query, $startDate, $endDate): Builder
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    /**
     * Get available years for filtering
     */
    public static function getAvailableYears()
    {
        return static::selectRaw('YEAR(created_at) as year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year');
    }

    /**
     * Get available months for filtering
     */
    public static function getAvailableMonths()
    {
        $months = static::selectRaw('MONTH(created_at) as month')
            ->distinct()
            ->orderBy('month')
            ->pluck('month')
            ->toArray();
            
        $monthNames = [
            1 => 'January', 2 => 'February', 3 => 'March', 4 => 'April',
            5 => 'May', 6 => 'June', 7 => 'July', 8 => 'August',
            9 => 'September', 10 => 'October', 11 => 'November', 12 => 'December'
        ];
        
        return collect($months)->mapWithKeys(function ($month) use ($monthNames) {
            return [$month => $monthNames[$month] ?? "Month {$month}"];
        });
    }
}

