<?php

namespace FedExShipping\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Ygs\CoreServices\Settings\Contracts\SettingsInterface;

class FedExService
{
    private string $baseUrl;
    private string $clientId;
    private string $clientSecret;
    private string $accountNumber;
    private bool $isActive;
    private ?string $accessToken = null;
    private ?int $tokenExpiresAt = null;
    protected SettingsInterface $settings;

    public function __construct(SettingsInterface $settings)
    {
        $this->settings = $settings;
        $this->baseUrl = $settings->get('fedex.base_url', 'https://apis-sandbox.fedex.com');
        $this->clientId = $settings->get('fedex.client_id', '');
        $this->clientSecret = $settings->get('fedex.client_secret', '');
        $this->accountNumber = $settings->get('fedex.account_number', '');
        $this->isActive = $settings->get('fedex.active', false);
    }

    /**
     * Check if FedEx integration is active and configured
     */
    public function isConfigured(): bool
    {
        return $this->isActive && 
               !empty($this->clientId) && 
               !empty($this->clientSecret) && 
               !empty($this->accountNumber);
    }

    /**
     * Get OAuth access token for FedEx API
     */
    private function getAccessToken(): ?string
    {
        // Check if we have a valid token
        if ($this->accessToken && $this->tokenExpiresAt && time() < $this->tokenExpiresAt) {
            return $this->accessToken;
        }

        try {
            $response = Http::asForm()->post($this->baseUrl . '/oauth/token', [
                'grant_type' => 'client_credentials',
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $this->accessToken = $data['access_token'];
                $this->tokenExpiresAt = time() + ($data['expires_in'] - 60); // Subtract 60 seconds for safety
                
                Log::info('FedEx access token obtained successfully');
                return $this->accessToken;
            } else {
                Log::error('Failed to get FedEx access token', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Exception getting FedEx access token', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Validate a single address using FedEx Address Validation API
     */
    public function validateAddress(array $address): array
    {
        if (!$this->isConfigured()) {
            return [
                'success' => false,
                'error' => 'FedEx integration is not configured or active'
            ];
        }

        // Only support US addresses for now
        if (($address['country_code'] ?? '') !== 'US') {
            return [
                'success' => false,
                'error' => 'Only US addresses are currently supported for FedEx validation'
            ];
        }

        // Generate cache key for this address
        $cacheKey = $this->generateAddressCacheKey($address);
        
        // Check if we have a cached result
        $cachedResult = Cache::get($cacheKey);
        if ($cachedResult !== null) {
            Log::info('FedEx address validation: Using cached result', [
                'address' => $address,
                'cache_key' => $cacheKey
            ]);
            return $cachedResult;
        }

        $accessToken = $this->getAccessToken();
        if (!$accessToken) {
            return [
                'success' => false,
                'error' => 'Failed to obtain FedEx access token'
            ];
        }

        try {
            // Format postal code for US addresses (ensure it's 5 digits)
            $postalCode = $address['postcode'] ?? '';
            if ($address['country_code'] === 'US' && strlen($postalCode) === 5) {
                // Ensure it's a 5-digit US postal code
                $postalCode = substr($postalCode, 0, 5);
            }
            
            $payload = [
                'addressesToValidate' => [
                    [
                        'address' => [
                            'streetLines' => array_filter([
                                $address['line_one'] ?? '',
                                $address['line_two'] ?? '',
                                $address['line_three'] ?? ''
                            ]),
                            'city' => $address['city'] ?? '',
                            'stateOrProvinceCode' => $address['state'] ?? '',
                            'postalCode' => $postalCode,
                            'countryCode' => $address['country_code'] ?? 'US'
                        ]
                    ]
                ]
            ];
            
            // Log the exact payload being sent
            Log::info('FedEx address validation payload', [
                'payload' => $payload,
                'address' => $address
            ]);

            $response = Http::timeout(45)->withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json',
                'X-locale' => 'en_US',
                'Accept-Language' => 'en-US'
            ])->post($this->baseUrl . '/address/v1/addresses/resolve', $payload);
            
            // Log the response for debugging
            Log::info('FedEx API response', [
                'status' => $response->status(),
                'response' => $response->json(),
                'payload' => $payload,
                'base_url' => $this->baseUrl
            ]);

            if ($response->successful()) {
                $data = $response->json();
                
                Log::info('FedEx address validation successful', [
                    'address' => $address,
                    'response' => $data
                ]);

                $result = [
                    'success' => true,
                    'data' => $data,
                    'validated_address' => $this->parseValidationResponse($data)
                ];

                // Only cache successful results with valid addresses
                $validatedAddress = $result['validated_address'];
                if ($validatedAddress['is_valid'] && empty($validatedAddress['errors'])) {
                    // Cache for 24 hours (1440 minutes)
                    Cache::put($cacheKey, $result, now()->addHours(24));
                    Log::info('FedEx address validation: Cached successful result', [
                        'address' => $address,
                        'cache_key' => $cacheKey,
                        'cache_duration' => '24 hours'
                    ]);
                }

                return $result;
            } else {
                Log::error('FedEx address validation failed', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'address' => $address
                ]);

                return [
                    'success' => false,
                    'error' => 'FedEx API request failed: ' . $response->status(),
                    'details' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception during FedEx address validation', [
                'error' => $e->getMessage(),
                'address' => $address
            ]);

            return [
                'success' => false,
                'error' => 'Exception during address validation: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Test the FedEx integration with a sample address
     */
    public function testConnection(): array
    {
        $testAddress = [
            'line_one' => '3650 W Market St',
            'city' => 'York',
            'state' => 'PA',
            'postcode' => '17404',
            'country_code' => 'US'
        ];

        return $this->validateAddress($testAddress);
    }

    /**
     * Parse the FedEx validation response into a standardized format
     */
    private function parseValidationResponse(array $response): array
    {
        $result = [
            'is_valid' => false,
            'confidence' => 'UNKNOWN',
            'suggestions' => [],
            'errors' => [],
            'warnings' => []
        ];

        if (empty($response['output'])) {
            return $result;
        }

        $output = $response['output'][0] ?? null;
        if (!$output) {
            return $result;
        }

        // Check if address is valid
        $result['is_valid'] = ($output['resolved'] ?? false);

        // Get confidence level
        if (isset($output['attributes'])) {
            foreach ($output['attributes'] as $attribute) {
                if ($attribute['name'] === 'ADDRESS_PRECISION') {
                    $result['confidence'] = $attribute['value'];
                    break;
                }
            }
        }

        // Get customer messages (errors/warnings)
        if (isset($output['customerMessages'])) {
            foreach ($output['customerMessages'] as $message) {
                $messageType = $message['severity'] ?? 'INFO';
                $messageText = $message['message'] ?? '';
                
                if ($messageType === 'ERROR') {
                    $result['errors'][] = $messageText;
                } elseif ($messageType === 'WARNING') {
                    $result['warnings'][] = $messageText;
                }
            }
        }

        // Get suggested addresses
        if (isset($output['resolvedAddresses'])) {
            foreach ($output['resolvedAddresses'] as $resolvedAddress) {
                $suggestion = [
                    'street_lines' => $resolvedAddress['streetLines'] ?? [],
                    'city' => $resolvedAddress['city'] ?? '',
                    'state' => $resolvedAddress['stateOrProvinceCode'] ?? '',
                    'postal_code' => $resolvedAddress['postalCode'] ?? '',
                    'country_code' => $resolvedAddress['countryCode'] ?? '',
                    'residential' => $resolvedAddress['residential'] ?? false
                ];
                
                $result['suggestions'][] = $suggestion;
            }
        }

        return $result;
    }

    /**
     * Generate a cache key for an address
     */
    private function generateAddressCacheKey(array $address): string
    {
        // Normalize address data for consistent cache keys
        $normalizedAddress = [
            'line_one' => strtolower(trim($address['line_one'] ?? '')),
            'line_two' => strtolower(trim($address['line_two'] ?? '')),
            'city' => strtolower(trim($address['city'] ?? '')),
            'state' => strtoupper(trim($address['state'] ?? '')),
            'postcode' => trim($address['postcode'] ?? ''),
            'country_code' => strtoupper(trim($address['country_code'] ?? '')),
        ];

        // Create a hash of the normalized address
        $addressHash = md5(json_encode($normalizedAddress));
        
        return "fedex_address_validation_{$addressHash}";
    }

    /**
     * Clear all cached address validation results
     */
    public function clearAddressCache(): void
    {
        $cache = Cache::store();
        $clearedCount = 0;
        
        if ($cache instanceof \Illuminate\Cache\RedisStore) {
            $redis = $cache->getRedis();
            $keys = $redis->keys('*fedex_address_validation_*');
            if (!empty($keys)) {
                $redis->del($keys);
                $clearedCount = count($keys);
            }
        } elseif ($cache instanceof \Illuminate\Cache\FileStore) {
            $filesystem = $cache->getFilesystem();
            $directory = $cache->getDirectory();
            $files = $filesystem->allFiles($directory);
            
            foreach ($files as $file) {
                $filename = basename($file);
                if (str_contains($filename, 'fedex_address_validation_')) {
                    $filesystem->delete($file);
                    $clearedCount++;
                }
            }
        } elseif ($cache instanceof \Illuminate\Cache\DatabaseStore) {
            $table = $cache->getTable();
            $connection = $cache->getConnection();
            $clearedCount = $connection->table($table)
                ->where('key', 'like', '%fedex_address_validation_%')
                ->delete();
        }
        
        Log::info("FedEx address validation cache: Cleared {$clearedCount} entries");
    }
} 