<?php

namespace FedExShipping\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Ygs\CoreServices\Settings\Contracts\SettingsInterface;
use Lunar\Models\Product;
use Lunar\Models\ProductVariant;
use Cartalyst\Converter\Laravel\Facades\Converter;

class FedExQuoteCalculator
{
    private string $baseUrl;
    private string $clientId;
    private string $clientSecret;
    private string $accountNumber;
    private ?string $accessToken = null;
    protected SettingsInterface $settings;

    // FedEx Service Types
    const SERVICE_TYPES = [
        'FIRST_OVERNIGHT' => 'First Overnight',
        'PRIORITY_OVERNIGHT' => 'Priority Overnight', 
        'STANDARD_OVERNIGHT' => 'Standard Overnight',
        'FEDEX_2_DAY_AM' => 'FedEx 2 Day AM',
        'FEDEX_2_DAY' => 'FedEx 2 Day',
        'FEDEX_EXPRESS_SAVER' => 'FedEx Express Saver',
        'GROUND_HOME_DELIVERY' => 'Ground Home Delivery',
        'FEDEX_GROUND' => 'FedEx Ground',
        'FEDEX_TUBE' => 'FedEx Tube',
        'FEDEX_FIRST_FREIGHT' => 'FedEx First Freight',
        'FEDEX_FREIGHT_ECONOMY' => 'FedEx Freight Economy',
        'FEDEX_FREIGHT_PRIORITY' => 'FedEx Freight Priority',
        'FEDEX_ONE_RATE' => 'FedEx One Rate',
    ];

    public function __construct(SettingsInterface $settings)
    {
        $this->settings = $settings;
        $this->baseUrl = $settings->get('fedex.base_url', 'https://apis-sandbox.fedex.com');
        $this->clientId = $settings->get('fedex.client_id', '');
        $this->clientSecret = $settings->get('fedex.client_secret', '');
        $this->accountNumber = $settings->get('fedex.account_number', '');
    }

    /**
     * Get shipping quotes for a product to a destination
     */
    public function getQuotes(Product $product, array $destinationAddress, array $options = []): array
    {
        try {
            // Get product dimensions and weight
            $variant = $product->variants()->first();
            if (!$variant) {
                throw new \Exception('Product has no variants');
            }

            // Generate cache key for this quote request
            $cacheKey = $this->generateQuoteCacheKey($product, $destinationAddress, $options);
            
            // Check if we have a cached result
            $cachedResult = Cache::get($cacheKey);
            if ($cachedResult !== null) {
                Log::info('FedEx quote calculator: Using cached result', [
                    'product_id' => $product->id,
                    'cache_key' => $cacheKey
                ]);
                return $cachedResult;
            }

            // Get sender address from settings
            $senderAddress = $this->getSenderAddress();
            
            // Prepare the request payload
            $payload = $this->buildQuoteRequest(
                $senderAddress,
                $destinationAddress,
                $variant,
                $options
            );

            // Make the API request
            $response = $this->makeApiRequest('/rate/v1/rates/quotes', $payload);
            
            $result = $this->parseQuoteResponse($response);
            
            // Only cache successful results with quotes
            if ($result['success'] && !empty($result['quotes'])) {
                // Cache for 1 hour (rates can change frequently)
                Cache::put($cacheKey, $result, now()->addHour());
                Log::info('FedEx quote calculator: Cached successful result', [
                    'product_id' => $product->id,
                    'cache_key' => $cacheKey,
                    'cache_duration' => '1 hour',
                    'quote_count' => count($result['quotes'])
                ]);
            }
            
            return $result;
            
        } catch (\Exception $e) {
            Log::error('FedEx Quote Calculator Error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'quotes' => []
            ];
        }
    }

    /**
     * Get shipping quotes for multiple products
     */
    public function getBulkQuotes(array $productIds, array $destinationAddress, array $options = []): array
    {
        // Safety limit: Maximum 50 products for bulk quotes to prevent excessive processing
        $maxProducts = 50;
        if (count($productIds) > $maxProducts) {
            Log::warning('FedExQuoteCalculator: Too many products for bulk quotes', [
                'product_count' => count($productIds),
                'max_allowed' => $maxProducts
            ]);
            $productIds = array_slice($productIds, 0, $maxProducts);
        }
        
        $results = [];
        $processedCount = 0;
        
        foreach ($productIds as $productId) {
            $product = Product::find($productId);
            if ($product) {
                $results[$productId] = $this->getQuotes($product, $destinationAddress, $options);
                $processedCount++;
            }
        }
        
        Log::info('FedExQuoteCalculator: Bulk quotes completed', [
            'requested_products' => count($productIds),
            'processed_products' => $processedCount,
            'successful_quotes' => count(array_filter($results, fn($r) => $r['success']))
        ]);
        
        return $results;
    }

    /**
     * Get sender address from settings
     */
    private function getSenderAddress(): array
    {
        $streetLines = $this->settings->get('sender.street_lines', ['']);
        $streetLines = is_array($streetLines) ? $streetLines : [$streetLines];
        
        return [
            'streetLines' => array_filter($streetLines), // Remove empty lines
            'city' => $this->settings->get('sender.city', ''),
            'stateOrProvinceCode' => $this->settings->get('sender.state_or_province_code', ''),
            'postalCode' => $this->settings->get('sender.postal_code', ''),
            'countryCode' => $this->settings->get('sender.country_code', 'US'),
            'residential' => $this->settings->get('sender.residential', false),
        ];
    }

    /**
     * Validate and format destination address
     */
    private function formatDestinationAddress(array $address): array
    {
        // Handle both snake_case and camelCase keys
        $streetLines = [];
        if (isset($address['street_lines'])) {
            $streetLines = is_array($address['street_lines']) ? $address['street_lines'] : [$address['street_lines']];
        } elseif (isset($address['streetLines'])) {
            $streetLines = is_array($address['streetLines']) ? $address['streetLines'] : [$address['streetLines']];
        }
        
        // Filter out empty values
        $streetLines = array_filter($streetLines);
        
        // Ensure we have at least one street line
        if (empty($streetLines)) {
            $streetLines = [''];
        }
        
        return [
            'streetLines' => $streetLines,
            'city' => $address['city'] ?? '',
            'stateOrProvinceCode' => $address['state_or_province_code'] ?? $address['stateOrProvinceCode'] ?? '',
            'postalCode' => $address['postal_code'] ?? $address['postalCode'] ?? '',
            'countryCode' => $address['country_code'] ?? $address['countryCode'] ?? 'US',
            'residential' => $address['residential'] ?? false,
        ];
    }

    /**
     * Build the quote request payload
     */
    private function buildQuoteRequest(array $senderAddress, array $destinationAddress, ProductVariant $variant, array $options = []): array
    {
        // Convert dimensions to inches if needed
        $dimensions = $this->convertDimensionsToInches($variant);
        
        // Convert weight to pounds if needed
        $weight = $this->convertWeightToPounds($variant);

        // Debug: Log weight and dimensions
        \Log::info('Product dimensions and weight:', [
            'product_id' => $variant->product_id,
            'variant_id' => $variant->id,
            'original_weight' => ($variant->weight_value ?? 'null') . ' ' . ($variant->weight_unit ?? 'null'),
            'converted_weight' => $weight . ' lb',
            'original_dimensions' => [
                'length' => ($variant->length_value ?? 'null') . ' ' . ($variant->length_unit ?? 'null'),
                'width' => ($variant->width_value ?? 'null') . ' ' . ($variant->width_unit ?? 'null'),
                'height' => ($variant->height_value ?? 'null') . ' ' . ($variant->height_unit ?? 'null'),
            ],
            'converted_dimensions' => $dimensions,
        ]);

        // Validate weight and dimensions
        if ($weight <= 0) {
            throw new \Exception('Invalid weight: ' . $weight . ' lb. Weight must be greater than 0.');
        }
        
        if ($dimensions['length'] <= 0 || $dimensions['width'] <= 0 || $dimensions['height'] <= 0) {
            throw new \Exception('Invalid dimensions: ' . json_encode($dimensions) . '. All dimensions must be greater than 0.');
        }

        // Format addresses properly
        $formattedSenderAddress = $this->formatDestinationAddress($senderAddress);
        $formattedDestinationAddress = $this->formatDestinationAddress($destinationAddress);

        $payload = [
            'accountNumber' => [
                'value' => $this->accountNumber
            ],

            'requestedShipment' => [
                'shipper' => [
                    'address' => $formattedSenderAddress
                ],
                'recipient' => [
                    'address' => $formattedDestinationAddress
                ],
                'preferredCurrency' => 'USD',
                'rateRequestType' => ['ACCOUNT', 'LIST'],
                'shipDateStamp' => now()->format('Y-m-d'),
                'pickupType' => 'DROPOFF_AT_FEDEX_LOCATION',
                'requestedPackageLineItems' => [
                    [
                        'weight' => [
                            'units' => 'LB',
                            'value' => $weight
                        ],
                        'dimensions' => [
                            'length' => $dimensions['length'],
                            'width' => $dimensions['width'],
                            'height' => $dimensions['height'],
                            'units' => 'IN'
                        ]
                    ]
                ],
                'packagingType' => 'YOUR_PACKAGING',
                'totalPackageCount' => 1,
                'totalWeight' => $weight
            ]
        ];

        // Add specific service types if requested
        if (!empty($options['service_types'])) {
            $payload['requestedShipment']['serviceType'] = $options['service_types'];
        }

        return $payload;
    }

    /**
     * Convert product dimensions to inches
     */
    private function convertDimensionsToInches(ProductVariant $variant): array
    {
        // Get dimensions with fallbacks
        $length = $variant->length_value ?? 10; // Default 10 inches
        $width = $variant->width_value ?? 8;   // Default 8 inches  
        $height = $variant->height_value ?? 2;  // Default 2 inches
        
        $lengthUnit = $variant->length_unit ?? 'in';
        $widthUnit = $variant->width_unit ?? 'in';
        $heightUnit = $variant->height_unit ?? 'in';
        
        // Convert to inches if needed
        $lengthInches = $this->convertToInches($length, $lengthUnit);
        $widthInches = $this->convertToInches($width, $widthUnit);
        $heightInches = $this->convertToInches($height, $heightUnit);
        
        // Ensure minimum dimensions (FedEx requirements)
        $lengthInches = max(0.1, $lengthInches);
        $widthInches = max(0.1, $widthInches);
        $heightInches = max(0.1, $heightInches);

        return [
            'length' => round($lengthInches, 2),
            'width' => round($widthInches, 2),
            'height' => round($heightInches, 2)
        ];
    }

    /**
     * Convert weight to pounds
     */
    private function convertWeightToPounds(ProductVariant $variant): float
    {
        // Get weight with fallback
        $weight = $variant->weight_value ?? 1; // Default 1 pound
        $weightUnit = $variant->weight_unit ?? 'lb';
        
        // Convert to pounds if needed
        $weightPounds = $this->convertToPounds($weight, $weightUnit);
        
        // Ensure minimum weight (FedEx requirements)
        $weightPounds = max(0.1, $weightPounds);
        
        return round($weightPounds, 2);
    }

    /**
     * Convert a measurement to inches
     */
    private function convertToInches(float $value, string $fromUnit): float
    {
        if ($fromUnit === 'in' || $fromUnit === 'inch') {
            return $value;
        }

        try {
            $converter = app('converter');
            return $converter->from($fromUnit)->to('in')->convert($value);
        } catch (\Exception $e) {
            // Fallback conversion for common units
            return $this->fallbackConversionToInches($value, $fromUnit);
        }
    }

    /**
     * Convert a weight to pounds
     */
    private function convertToPounds(float $value, string $fromUnit): float
    {
        if ($fromUnit === 'lb' || $fromUnit === 'lbs') {
            return $value;
        }

        try {
            $converter = app('converter');
            return $converter->from($fromUnit)->to('lb')->convert($value);
        } catch (\Exception $e) {
            // Fallback conversion for common units
            return $this->fallbackConversionToPounds($value, $fromUnit);
        }
    }

    /**
     * Fallback conversion to inches for common units
     */
    private function fallbackConversionToInches(float $value, string $fromUnit): float
    {
        $conversions = [
            'cm' => 0.393701,
            'mm' => 0.0393701,
            'm' => 39.3701,
            'ft' => 12,
            'yd' => 36,
        ];

        return $value * ($conversions[$fromUnit] ?? 1);
    }

    /**
     * Fallback conversion to pounds for common units
     */
    private function fallbackConversionToPounds(float $value, string $fromUnit): float
    {
        $conversions = [
            'kg' => 2.20462,
            'g' => 0.00220462,
            'oz' => 0.0625,
        ];

        return $value * ($conversions[$fromUnit] ?? 1);
    }

    /**
     * Make API request to FedEx
     */
    private function makeApiRequest(string $endpoint, array $payload): array
    {
        $token = $this->getAccessToken();
        
        // Safety timeout: 30 seconds to prevent hanging requests
        $response = Http::timeout(45)->withHeaders([
            'Authorization' => 'Bearer ' . $token,
            'X-locale' => 'en_US',
            'Content-Type' => 'application/json'
        ])->post($this->baseUrl . $endpoint, $payload);

        if (!$response->successful()) {
            throw new \Exception('FedEx API Error: ' . $response->body());
        }

        return $response->json();
    }

    /**
     * Get or refresh access token
     */
    private function getAccessToken(): string
    {
        if ($this->accessToken) {
            return $this->accessToken;
        }

        $response = Http::asForm()->post($this->baseUrl . '/oauth/token', [
            'grant_type' => 'client_credentials',
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
        ]);

        if (!$response->successful()) {
            throw new \Exception('Failed to get FedEx access token: ' . $response->body());
        }

        $data = $response->json();
        $this->accessToken = $data['access_token'];

        return $this->accessToken;
    }

    /**
     * Parse the quote response
     */
    private function parseQuoteResponse(array $response): array
    {
        if (!isset($response['output']['rateReplyDetails'])) {
            return [
                'success' => false,
                'error' => 'No rate details found in response',
                'quotes' => []
            ];
        }

        $quotes = [];
        $enabledServices = $this->getEnabledServices();
        
        // Safety limit: Maximum 30 rate details to prevent excessive processing
        $maxRateDetails = 30;
        $rateDetails = $response['output']['rateReplyDetails'];
        if (count($rateDetails) > $maxRateDetails) {
            Log::warning('FedExQuoteCalculator: Too many rate details in response', [
                'rate_details_count' => count($rateDetails),
                'max_allowed' => $maxRateDetails
            ]);
            $rateDetails = array_slice($rateDetails, 0, $maxRateDetails);
        }
        
        foreach ($rateDetails as $rateDetail) {
            // Get the first rated shipment detail (usually the ACCOUNT rate)
            $ratedShipment = $rateDetail['ratedShipmentDetails'][0] ?? null;
            
            if (!$ratedShipment) {
                continue;
            }
            
            $serviceType = $rateDetail['serviceType'] ?? '';
            
            // Filter by enabled services if any are configured
            if (!empty($enabledServices) && !in_array($serviceType, $enabledServices)) {
                continue;
            }
            
            $quotes[] = [
                'service_type' => $serviceType,
                'service_name' => $rateDetail['serviceName'] ?? '',
                'total_charge' => $ratedShipment['totalNetCharge'] ?? 0,
                'currency' => $ratedShipment['currency'] ?? 'USD',
                'delivery_date' => $rateDetail['deliveryDate'] ?? null,
                'transit_time' => $rateDetail['transitTime'] ?? null,
                'rate_type' => $ratedShipment['rateType'] ?? '',
                'base_charge' => $ratedShipment['totalBaseCharge'] ?? 0,
                'discounts' => $ratedShipment['totalDiscounts'] ?? 0,
            ];
        }

        return [
            'success' => true,
            'quotes' => $quotes
        ];
    }

    /**
     * Get enabled services from settings
     */
    private function getEnabledServices(): array
    {
        $enabledServices = $this->settings->get('fedex.enabled_services', []);
        
        if (is_string($enabledServices)) {
            $enabledServices = json_decode($enabledServices, true);
        }
        
        return is_array($enabledServices) ? $enabledServices : [];
    }

    /**
     * Get available service types
     */
    public static function getServiceTypes(): array
    {
        return self::SERVICE_TYPES;
    }

    /**
     * Get enabled service types for frontend display
     * Note: This is a static method, so it cannot use dependency injection.
     * For plugin decoupling, this should be refactored to an instance method
     * or use app() helper to resolve SettingsInterface.
     */
    public static function getEnabledServiceTypes(): array
    {
        $settings = app(\Ygs\CoreServices\Settings\Contracts\SettingsInterface::class);
        $enabledServices = $settings->get('fedex.enabled_services', []);
        
        if (is_string($enabledServices)) {
            $enabledServices = json_decode($enabledServices, true);
        }
        
        if (!is_array($enabledServices) || empty($enabledServices)) {
            // If no services are specifically enabled, return all available
            return self::SERVICE_TYPES;
        }
        
        // Return only the enabled services
        $enabledTypes = [];
        foreach ($enabledServices as $serviceKey) {
            if (isset(self::SERVICE_TYPES[$serviceKey])) {
                $enabledTypes[$serviceKey] = self::SERVICE_TYPES[$serviceKey];
            }
        }
        
        return $enabledTypes;
    }

    /**
     * Validate if FedEx is configured
     */
    public function isConfigured(): bool
    {
        return !empty($this->clientId) && 
               !empty($this->clientSecret) && 
               !empty($this->accountNumber);
    }

    /**
     * Generate a cache key for a quote request
     */
    private function generateQuoteCacheKey(Product $product, array $destinationAddress, array $options = []): string
    {
        // Get product variant for dimensions and weight
        $variant = $product->variants()->first();
        if (!$variant) {
            throw new \Exception('Product has no variants');
        }

        // Normalize address data for consistent cache keys
        $normalizedAddress = [
            'line_one' => strtolower(trim($destinationAddress['line_one'] ?? '')),
            'line_two' => strtolower(trim($destinationAddress['line_two'] ?? '')),
            'city' => strtolower(trim($destinationAddress['city'] ?? '')),
            'state' => strtoupper(trim($destinationAddress['state'] ?? '')),
            'postcode' => trim($destinationAddress['postal_code'] ?? ''),
            'country_code' => strtoupper(trim($destinationAddress['country_code'] ?? '')),
        ];

        // Create a hash of the product, address, and options
        $cacheData = [
            'product_id' => $product->id,
            'variant_id' => $variant->id,
            'weight' => $variant->weight_value,
            'weight_unit' => $variant->weight_unit,
            'length' => $variant->length_value,
            'width' => $variant->width_value,
            'height' => $variant->height_value,
            'length_unit' => $variant->length_unit,
            'address' => $normalizedAddress,
            'options' => $options,
        ];

        $dataHash = md5(json_encode($cacheData));
        
        return "fedex_quote_{$dataHash}";
    }

    /**
     * Clear all cached quote results
     */
    public function clearQuoteCache(): void
    {
        $cache = Cache::store();
        $clearedCount = 0;
        
        if ($cache instanceof \Illuminate\Cache\RedisStore) {
            $redis = $cache->getRedis();
            $keys = $redis->keys('*fedex_quote_*');
            if (!empty($keys)) {
                $redis->del($keys);
                $clearedCount = count($keys);
            }
        } elseif ($cache instanceof \Illuminate\Cache\FileStore) {
            $filesystem = $cache->getFilesystem();
            $directory = $cache->getDirectory();
            $files = $filesystem->allFiles($directory);
            
            foreach ($files as $file) {
                $filename = basename($file);
                if (str_contains($filename, 'fedex_quote_')) {
                    $filesystem->delete($file);
                    $clearedCount++;
                }
            }
        } elseif ($cache instanceof \Illuminate\Cache\DatabaseStore) {
            $table = $cache->getTable();
            $connection = $cache->getConnection();
            $clearedCount = $connection->table($table)
                ->where('key', 'like', '%fedex_quote_%')
                ->delete();
        }
        
        Log::info("FedEx quote calculator cache: Cleared {$clearedCount} entries");
    }
} 