<?php

namespace FedExShipping\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class FedExCacheManager
{
    /**
     * Clear all FedEx-related cache entries
     */
    public static function clearAllCache(): array
    {
        $results = [
            'address_validation' => 0,
            'quotes' => 0,
            'total' => 0
        ];

        // Clear address validation cache
        $results['address_validation'] = self::clearAddressValidationCache();
        
        // Clear quote cache
        $results['quotes'] = self::clearQuoteCache();
        
        $results['total'] = $results['address_validation'] + $results['quotes'];
        
        Log::info('FedEx cache manager: Cleared all cache', $results);
        
        return $results;
    }

    /**
     * Clear address validation cache
     */
    public static function clearAddressValidationCache(): int
    {
        return self::clearCacheByPattern('fedex_address_validation_');
    }

    /**
     * Clear quote cache
     */
    public static function clearQuoteCache(): int
    {
        return self::clearCacheByPattern('fedex_quote_');
    }

    /**
     * Clear cache entries by pattern
     */
    private static function clearCacheByPattern(string $pattern): int
    {
        $cache = Cache::store();
        $clearedCount = 0;
        
        try {
            if ($cache instanceof \Illuminate\Cache\RedisStore) {
                $clearedCount = self::clearRedisCache($cache, $pattern);
            } elseif ($cache instanceof \Illuminate\Cache\FileStore) {
                $clearedCount = self::clearFileCache($cache, $pattern);
            } elseif ($cache instanceof \Illuminate\Cache\DatabaseStore) {
                $clearedCount = self::clearDatabaseCache($cache, $pattern);
            } else {
                Log::warning("FedEx cache manager: Unsupported cache driver for pattern clearing");
                return 0;
            }
        } catch (\Exception $e) {
            Log::error("FedEx cache manager: Error clearing cache with pattern '{$pattern}': " . $e->getMessage());
            return 0;
        }
        
        return $clearedCount;
    }

    /**
     * Clear Redis cache by pattern
     */
    private static function clearRedisCache($cache, string $pattern): int
    {
        $redis = $cache->getRedis();
        $keys = $redis->keys("*{$pattern}*");
        
        // Safety limit: Maximum 1000 cache keys to prevent excessive deletion
        $maxKeys = 1000;
        if (count($keys) > $maxKeys) {
            Log::warning("FedEx cache manager: Too many cache keys to clear", [
                'pattern' => $pattern,
                'key_count' => count($keys),
                'max_allowed' => $maxKeys
            ]);
            $keys = array_slice($keys, 0, $maxKeys);
        }
        
        if (!empty($keys)) {
            $redis->del($keys);
        }
        
        return count($keys);
    }

    /**
     * Clear file cache by pattern
     */
    private static function clearFileCache($cache, string $pattern): int
    {
        $filesystem = $cache->getFilesystem();
        $directory = $cache->getDirectory();
        
        $clearedCount = 0;
        $files = $filesystem->allFiles($directory);
        
        // Safety limit: Maximum 500 files to prevent excessive file operations
        $maxFiles = 500;
        if (count($files) > $maxFiles) {
            Log::warning("FedEx cache manager: Too many cache files to process", [
                'pattern' => $pattern,
                'file_count' => count($files),
                'max_allowed' => $maxFiles
            ]);
            $files = array_slice($files, 0, $maxFiles);
        }
        
        foreach ($files as $file) {
            $filename = basename($file);
            if (str_contains($filename, $pattern)) {
                $filesystem->delete($file);
                $clearedCount++;
            }
        }
        
        return $clearedCount;
    }

    /**
     * Clear database cache by pattern
     */
    private static function clearDatabaseCache($cache, string $pattern): int
    {
        $table = $cache->getTable();
        $connection = $cache->getConnection();
        
        $clearedCount = $connection->table($table)
            ->where('key', 'like', "%{$pattern}%")
            ->delete();
        
        return $clearedCount;
    }

    /**
     * Get cache statistics
     */
    public static function getCacheStats(): array
    {
        $stats = [
            'address_validation_count' => 0,
            'quotes_count' => 0,
            'total_count' => 0,
            'cache_driver' => config('cache.default')
        ];

        try {
            $cache = Cache::store();
            
            if ($cache instanceof \Illuminate\Cache\RedisStore) {
                $redis = $cache->getRedis();
                $addressKeys = $redis->keys('*fedex_address_validation_*');
                $quoteKeys = $redis->keys('*fedex_quote_*');
                
                $stats['address_validation_count'] = count($addressKeys);
                $stats['quotes_count'] = count($quoteKeys);
                $stats['total_count'] = $stats['address_validation_count'] + $stats['quotes_count'];
            }
        } catch (\Exception $e) {
            Log::error("FedEx cache manager: Error getting cache stats: " . $e->getMessage());
        }
        
        return $stats;
    }
} 