<?php

namespace FedExShipping;

use Ygs\CoreServices\Facades\HookManager;
use FedExShipping\Modifiers\FedExShippingModifier;
use Lunar\Base\ShippingModifiers;
use Ygs\CoreServices\Settings\Contracts\SettingsInterface;

class ServiceProvider extends \Illuminate\Support\ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        // Register the shipping modifier when shipping modifiers are being registered
        HookManager::addAction('shipping.modifiers.register', function(ShippingModifiers $modifiers) {
            $modifiers->add(\FedExShipping\Modifiers\FedExShippingModifier::class);
        }, 10); // Priority 10 - runs after defaults

        // Register FedEx shipping method via hook (no direct model access)
        HookManager::addAction('shipping.methods.register', function() {
            HookManager::doAction('shipping.method.create', [
                'name' => 'fedex-shipping',
                'title' => 'FedEx Shipping',
                'type' => 'plugin',
                'plugin_name' => 'fedex-shipping',
                'enabled' => false,
                'sort_order' => 10,
                'description' => 'FedEx shipping integration with real-time rate calculation',
            ]);
        }, 10);

        // Heartbeat: Check and correct FedEx method on every admin page load
        HookManager::addAction('shipping.methods.heartbeat', function() {
            HookManager::doAction('shipping.method.create', [
                'name' => 'fedex-shipping',
                'title' => 'FedEx Shipping',
                'type' => 'plugin',
                'plugin_name' => 'fedex-shipping',
                'enabled' => false,
                'sort_order' => 10,
                'description' => 'FedEx shipping integration with real-time rate calculation',
            ]);
        }, 10);

        // Add FedEx configuration form fields
        HookManager::addAction('shipping.method.form', function($record, &$schema) {
            if (!$record || (method_exists($record, 'getName') && $record->getName() !== 'fedex-shipping')) {
                return;
            }
            
            // Get settings interface from container
            $settings = app(SettingsInterface::class);
            
            $schema[] = \Filament\Forms\Components\Section::make('FedEx Configuration')
                ->description('Configure FedEx API credentials and settings')
                ->icon('heroicon-o-truck')
                ->schema([
                    \Filament\Forms\Components\Toggle::make('fedex_active')
                        ->label('Enable FedEx')
                        ->default(fn() => $settings->get('fedex.active', false))
                        ->helperText('Enable or disable FedEx shipping calculations'),
                    
                    \Filament\Forms\Components\TextInput::make('fedex_base_url')
                        ->label('FedEx API Base URL')
                        ->default(fn() => $settings->get('fedex.base_url', 'https://apis-sandbox.fedex.com'))
                        ->helperText('FedEx API base URL (use sandbox URL for testing)')
                        ->required(),
                    
                    \Filament\Forms\Components\TextInput::make('fedex_client_id')
                        ->label('Client ID')
                        ->default(fn() => $settings->get('fedex.client_id', ''))
                        ->helperText('Your FedEx API client ID')
                        ->required(),
                    
                    \Filament\Forms\Components\TextInput::make('fedex_client_secret')
                        ->label('Client Secret')
                        ->default(fn() => $settings->get('fedex.client_secret', ''))
                        ->helperText('Your FedEx API client secret')
                        ->password()
                        ->required(),
                    
                    \Filament\Forms\Components\TextInput::make('fedex_account_number')
                        ->label('Account Number')
                        ->default(fn() => $settings->get('fedex.account_number', ''))
                        ->helperText('Your FedEx account number')
                        ->required(),
                ]);
        }, 10);

        // Save FedEx configuration
        HookManager::addAction('shipping.method.save', function($record, $data) {
            if (!$record || (method_exists($record, 'getName') && $record->getName() !== 'fedex-shipping')) {
                return;
            }

            // Get settings interface from container
            $settings = app(SettingsInterface::class);

            // Save FedEx settings
            if (isset($data['fedex_active'])) {
                $settings->set('fedex.active', (bool)$data['fedex_active'], 'boolean', 'fedex');
            }
            if (isset($data['fedex_base_url'])) {
                $settings->set('fedex.base_url', $data['fedex_base_url'], 'string', 'fedex');
            }
            if (isset($data['fedex_client_id'])) {
                $settings->set('fedex.client_id', $data['fedex_client_id'], 'string', 'fedex');
            }
            if (isset($data['fedex_client_secret'])) {
                $settings->set('fedex.client_secret', $data['fedex_client_secret'], 'string', 'fedex');
            }
            if (isset($data['fedex_account_number'])) {
                $settings->set('fedex.account_number', $data['fedex_account_number'], 'string', 'fedex');
            }
        }, 10);

        // Register quote calculator for shipping calculator component
        HookManager::addAction('shipping.quote.calculate', function($quoteRequest, &$allQuotes) {
            // Get settings interface from container
            $settings = app(SettingsInterface::class);
            
            // Check if FedEx is enabled
            if (!$settings->get('fedex.active', false)) {
                return;
            }

            try {
                $quoteCalculator = new \FedExShipping\Services\FedExQuoteCalculator($settings);
                
                if (!$quoteCalculator->isConfigured()) {
                    return; // FedEx not configured, skip silently
                }

                $products = $quoteRequest['products'] ?? collect();
                $destinationAddress = $quoteRequest['destination_address'] ?? [];

                // Calculate quotes for each product
                foreach ($products as $product) {
                    $result = $quoteCalculator->getQuotes(
                        $product,
                        $destinationAddress,
                        []
                    );

                    if ($result['success'] && !empty($result['quotes'])) {
                        foreach ($result['quotes'] as $quote) {
                            $serviceKey = $quote['service_type'];
                            if (!isset($allQuotes[$serviceKey])) {
                                $allQuotes[$serviceKey] = [
                                    'service_type' => $quote['service_type'],
                                    'service_name' => $quote['service_name'],
                                    'rate_type' => $quote['rate_type'],
                                    'currency' => $quote['currency'],
                                    'transit_time' => $quote['transit_time'] ?? null,
                                    'delivery_date' => $quote['delivery_date'] ?? null,
                                    'total_charge' => 0,
                                    'base_charge' => 0,
                                    'discounts' => 0,
                                    'product_count' => 0,
                                ];
                            }
                            $allQuotes[$serviceKey]['total_charge'] += (float) ($quote['total_charge'] ?? 0);
                            $allQuotes[$serviceKey]['base_charge'] += (float) ($quote['base_charge'] ?? 0);
                            $allQuotes[$serviceKey]['discounts'] += (float) ($quote['discounts'] ?? 0);
                            $allQuotes[$serviceKey]['product_count']++;
                        }
                    }
                }
            } catch (\Exception $e) {
                // Log error but don't break other plugins' quote calculations
                \Log::error('FedEx quote calculation error: ' . $e->getMessage());
            }
        }, 10, 2); // Priority 10, accepts 2 arguments
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Load routes if they exist
        if (file_exists(__DIR__ . '/../routes/api.php')) {
            $this->loadRoutesFrom(__DIR__ . '/../routes/api.php');
        }

        // Load views if they exist
        if (is_dir(__DIR__ . '/../resources/views')) {
            $this->loadViewsFrom(__DIR__ . '/../resources/views', 'fedex-shipping');
        }
    }
}

