<?php

namespace FedExShipping\Modifiers;

use Lunar\Base\ShippingModifier;
use Lunar\DataTypes\Price;
use Lunar\DataTypes\ShippingOption;
use Lunar\Facades\ShippingManifest;
use Lunar\Models\Contracts\Cart;
use Lunar\Models\TaxClass;
use FedExShipping\Services\FedExQuoteCalculator;
use Ygs\CoreServices\Settings\Contracts\SettingsInterface;

class FedExShippingModifier extends ShippingModifier
{
    protected SettingsInterface $settings;

    public function __construct(SettingsInterface $settings)
    {
        $this->settings = $settings;
    }

    public function handle(Cart $cart, \Closure $next)
    {
        \Log::info('FedExShippingModifier: Starting', [
            'cart_id' => $cart->id,
            'has_lines' => !$cart->lines->isEmpty(),
            'has_shipping_address' => $cart->shippingAddress !== null
        ]);
        
        // Check if FedEx is enabled in global settings
        if (!$this->settings->get('fedex.active', false)) {
            \Log::info('FedExShippingModifier: FedEx not active in settings');
            return $next($cart);
        }

        // Check if cart has shippable items
        if ($cart->lines->isEmpty()) {
            \Log::info('FedExShippingModifier: Cart is empty');
            return $next($cart);
        }

        // Get shipping address from cart
        $shippingAddress = $cart->shippingAddress;
        if (!$shippingAddress) {
            \Log::info('FedExShippingModifier: No shipping address on cart');
            return $next($cart);
        }

        try {
            $quoteCalculator = new FedExQuoteCalculator($this->settings);
            
            if (!$quoteCalculator->isConfigured()) {
                return $next($cart);
            }

            // Convert Lunar address to our format
            $destinationAddress = [
                'street_lines' => [$shippingAddress->line_one],
                'city' => $shippingAddress->city,
                'state_or_province_code' => $shippingAddress->state,
                'postal_code' => $shippingAddress->postcode,
                'country_code' => $shippingAddress->country->iso2,
                'residential' => true, // Default to residential for B2C
            ];

            // Add line_two if it exists
            if ($shippingAddress->line_two) {
                $destinationAddress['street_lines'][] = $shippingAddress->line_two;
            }

            // Get all unique products from cart
            $products = $cart->lines->map(function ($line) {
                return $line->purchasable->product;
            })->unique('id');

            // Safety limit: Maximum 1000 unique products to prevent excessive API calls
            $maxProducts = 1000;
            if ($products->count() > $maxProducts) {
                \Log::warning('FedExShippingModifier: Too many unique products in cart', [
                    'product_count' => $products->count(),
                    'max_allowed' => $maxProducts,
                    'cart_id' => $cart->id
                ]);
                $products = $products->take($maxProducts);
            }

            // Calculate quotes for each product and aggregate
            $allQuotes = [];
            $processedProducts = 0;
            foreach ($products as $product) {
                $result = $quoteCalculator->getQuotes(
                    $product,
                    $destinationAddress,
                    []
                );

                if ($result['success'] && !empty($result['quotes'])) {
                    // Safety limit: Maximum 15 quotes per product to prevent excessive processing
                    $maxQuotesPerProduct = 15;
                    $quotesToProcess = array_slice($result['quotes'], 0, $maxQuotesPerProduct);
                    
                    foreach ($quotesToProcess as $quote) {
                        $serviceKey = $quote['service_type'];
                        if (!isset($allQuotes[$serviceKey])) {
                            $allQuotes[$serviceKey] = [
                                'service_type' => $quote['service_type'],
                                'service_name' => $quote['service_name'],
                                'rate_type' => $quote['rate_type'],
                                'currency' => $quote['currency'],
                                'transit_time' => $quote['transit_time'],
                                'delivery_date' => $quote['delivery_date'],
                                'total_charge' => 0,
                                'base_charge' => 0,
                                'discounts' => 0,
                                'product_count' => 0,
                            ];
                        }
                        $allQuotes[$serviceKey]['total_charge'] += (float) $quote['total_charge'];
                        $allQuotes[$serviceKey]['base_charge'] += (float) $quote['base_charge'];
                        $allQuotes[$serviceKey]['discounts'] += (float) $quote['discounts'];
                        $allQuotes[$serviceKey]['product_count']++;
                    }
                }
                
                $processedProducts++;
            }

            // Get the tax class
            $taxClass = TaxClass::first();

            \Log::info('FedExShippingModifier: Generated quotes', [
                'quote_count' => count($allQuotes),
                'processed_products' => $processedProducts,
                'quotes' => $allQuotes
            ]);
            
            // Safety limit: Maximum 25 total shipping options to prevent UI overload
            $maxShippingOptions = 25;
            if (count($allQuotes) > $maxShippingOptions) {
                \Log::warning('FedExShippingModifier: Too many shipping options generated', [
                    'option_count' => count($allQuotes),
                    'max_allowed' => $maxShippingOptions,
                    'cart_id' => $cart->id
                ]);
                // Keep only the first 25 options (usually the most relevant)
                $allQuotes = array_slice($allQuotes, 0, $maxShippingOptions, true);
            }
            
            // Add shipping options to manifest
            $addedOptions = 0;
            foreach ($allQuotes as $quote) {
                // Convert price to cents (Lunar uses cents)
                $priceInCents = (int) ($quote['total_charge'] * 100);
                
                // Create description with transit time if available
                $description = $quote['service_name'];
                if ($quote['transit_time']) {
                    $description .= ' - ' . $quote['transit_time'];
                }
                if ($quote['delivery_date']) {
                    $description .= ' (Est. delivery: ' . $quote['delivery_date'] . ')';
                }

                \Log::info('FedExShippingModifier: Adding shipping option', [
                    'name' => $quote['service_name'],
                    'identifier' => 'FEDEX_' . $quote['service_type'],
                    'price' => $priceInCents
                ]);

                ShippingManifest::addOption(
                    new ShippingOption(
                        name: $quote['service_name'],
                        description: $description,
                        identifier: 'FEDEX_' . $quote['service_type'],
                        price: new Price($priceInCents, $cart->currency, 1),
                        taxClass: $taxClass,
                        collect: false // FedEx handles collection
                    )
                );
                $addedOptions++;
            }
            
            \Log::info('FedExShippingModifier: Added shipping options to manifest', [
                'added_options' => $addedOptions,
                'cart_id' => $cart->id
            ]);

        } catch (\Exception $e) {
            // Log error but don't break checkout
            \Log::error('FedEx Shipping Modifier Error: ' . $e->getMessage(), [
                'cart_id' => $cart->id,
                'trace' => $e->getTraceAsString()
            ]);
        }

        return $next($cart);
    }
}

