<?php

namespace AuthNetPayment\Services;

use Ygs\CoreServices\Payment\Contracts\PaymentGatewayInterface;
use Ygs\CoreServices\Settings\Contracts\SettingsInterface;
use Lunar\Models\Order;
use Illuminate\Support\Facades\Log;
use net\authorize\api\contract\v1 as AnetAPI;
use net\authorize\api\controller as AnetController;

class AuthorizeNetGateway implements PaymentGatewayInterface
{
    private string $apiLoginId;
    private string $transactionKey;
    private string $environment;
    protected SettingsInterface $settings;

    public function __construct(SettingsInterface $settings)
    {
        $this->settings = $settings;
        $this->apiLoginId = $settings->get('authnet.api_login_id', '');
        $this->transactionKey = $settings->get('authnet.transaction_key', '');
        
        // Determine environment based on settings
        $baseUrl = $settings->get('authnet.base_url', 'https://apitest.authorize.net');
        $this->environment = (str_contains($baseUrl, 'sandbox') || str_contains($baseUrl, 'apitest')) 
            ? \net\authorize\api\constants\ANetEnvironment::SANDBOX 
            : \net\authorize\api\constants\ANetEnvironment::PRODUCTION;
    }

    /**
     * Get the payment gateway identifier
     */
    public function getIdentifier(): string
    {
        return 'authorize-net';
    }

    /**
     * Get the payment gateway display name
     */
    public function getName(): string
    {
        return 'Authorize.Net';
    }

    /**
     * Check if the payment gateway is configured and enabled
     */
    public function isConfigured(): bool
    {
        $isActive = $this->settings->get('authnet.active', false);
        $hasCredentials = !empty($this->apiLoginId) && !empty($this->transactionKey);
        
        return $isActive && $hasCredentials;
    }

    /**
     * Process a payment for an order
     */
    public function processPayment(Order $order, array $paymentData, array $billingAddress = []): array
    {
        try {
            // Extract payment data
            $dataDescriptor = $paymentData['data_descriptor'] ?? '';
            $dataValue = $paymentData['data_value'] ?? '';

            if (empty($dataDescriptor) || empty($dataValue)) {
                return [
                    'success' => false,
                    'error' => 'Missing payment token data'
                ];
            }

            // Create the payment data using opaque data from Accept.js
            $paymentOne = new AnetAPI\PaymentType();
            
            // Set the opaque data (payment token from Accept.js)
            $opaqueData = new AnetAPI\OpaqueDataType();
            $opaqueData->setDataDescriptor($dataDescriptor);
            $opaqueData->setDataValue($dataValue);
            
            $paymentOne->setOpaqueData($opaqueData);

            // Create a transaction
            $transactionRequestType = new AnetAPI\TransactionRequestType();
            $transactionRequestType->setTransactionType("authCaptureTransaction");
            $transactionRequestType->setPayment($paymentOne);
            $transactionRequestType->setAmount($order->total->value / 100); // Convert from cents to dollars

            // Set the customer's Bill To address
            if (!empty($billingAddress)) {
                $billTo = new AnetAPI\CustomerAddressType();
                $billTo->setFirstName($billingAddress['first_name'] ?? '');
                $billTo->setLastName($billingAddress['last_name'] ?? '');
                $billTo->setCompany($billingAddress['company'] ?? '');
                $billTo->setAddress($billingAddress['line_one'] ?? '');
                $billTo->setCity($billingAddress['city'] ?? '');
                $billTo->setState($billingAddress['state'] ?? '');
                $billTo->setZip($billingAddress['postcode'] ?? '');
                $billTo->setCountry($billingAddress['country_code'] ?? 'US');
                $billTo->setPhoneNumber($billingAddress['contact_phone'] ?? '');
                
                $transactionRequestType->setBillTo($billTo);
            }

            // Set the customer's Ship To address
            if ($order->shippingAddress) {
                $shipTo = new AnetAPI\CustomerAddressType();
                $shipTo->setFirstName($order->shippingAddress->first_name);
                $shipTo->setLastName($order->shippingAddress->last_name);
                $shipTo->setCompany($order->shippingAddress->company_name);
                $shipTo->setAddress($order->shippingAddress->line_one);
                $shipTo->setCity($order->shippingAddress->city);
                $shipTo->setState($order->shippingAddress->state);
                $shipTo->setZip($order->shippingAddress->postcode);
                $shipTo->setCountry($order->shippingAddress->country->iso2 ?? 'US');
                
                $transactionRequestType->setShipTo($shipTo);
            }

            // Set the customer's identifying information
            $customerData = new AnetAPI\CustomerDataType();
            $customerData->setType("individual");
            $customerData->setId($order->customer_id);
            $customerData->setEmail($order->shippingAddress->contact_email ?? '');
            
            $transactionRequestType->setCustomer($customerData);

            // Add line items
            $lineItems = [];
            foreach ($order->lines as $line) {
                $lineItem = new AnetAPI\LineItemType();
                $lineItem->setItemId($line->purchasable_id);
                $lineItem->setName($line->description);
                $lineItem->setQuantity($line->quantity);
                $lineItem->setUnitPrice($line->unit_price->value / 100);
                
                $lineItems[] = $lineItem;
            }
            $transactionRequestType->setLineItems($lineItems);

            // Assemble the complete transaction request
            $request = new AnetAPI\CreateTransactionRequest();
            $request->setMerchantAuthentication($this->getMerchantAuthentication());
            $request->setRefId($order->reference);
            $request->setTransactionRequest($transactionRequestType);

            // Log request details for debugging
            Log::info('Authorize.net payment request prepared', [
                'order_reference' => $order->reference,
                'amount' => $order->total->value / 100,
                'has_billing_address' => !empty($billingAddress),
                'has_shipping_address' => $order->shippingAddress !== null,
                'line_items_count' => count($lineItems),
                'environment' => $this->environment,
                'api_login_id' => $this->apiLoginId ? '***' . substr($this->apiLoginId, -4) : 'not_set'
            ]);

            // Create the controller and get the response
            $controller = new AnetController\CreateTransactionController($request);
            $response = $controller->executeWithApiResponse($this->environment);

            // Log the raw response for debugging
            Log::info('Authorize.net API response received', [
                'order_reference' => $order->reference,
                'response_type' => get_class($response),
                'has_transaction_response' => $response ? ($response->getTransactionResponse() !== null) : false,
                'has_messages' => $response ? ($response->getMessages() !== null) : false
            ]);

            if ($response != null) {
                $tresponse = $response->getTransactionResponse();

                if (($tresponse != null) && ($tresponse->getResponseCode() == "1")) {
                    // Success
                    Log::info('Authorize.net payment successful', [
                        'order_reference' => $order->reference,
                        'transaction_id' => $tresponse->getTransId(),
                        'auth_code' => $tresponse->getAuthCode(),
                        'response_code' => $tresponse->getResponseCode()
                    ]);

                    return [
                        'success' => true,
                        'transaction_id' => $tresponse->getTransId(),
                        'auth_code' => $tresponse->getAuthCode(),
                        'response_code' => $tresponse->getResponseCode(),
                        'message' => 'Payment processed successfully'
                    ];
                } else {
                    // Error - handle both null and non-null transaction response
                    $errorMessage = "Payment failed";
                    $responseCode = null;
                    
                    if ($tresponse != null) {
                        $responseCode = $tresponse->getResponseCode();
                        $errorMessages = $tresponse->getErrors();
                        
                        if (count($errorMessages) > 0) {
                            $errorMessage = $errorMessages[0]->getErrorText();
                        }
                    } else {
                        // Check for API-level errors
                        $apiErrors = $response->getMessages();
                        if ($apiErrors != null && count($apiErrors->getMessage()) > 0) {
                            $errorMessage = $apiErrors->getMessage()[0]->getText();
                        }
                    }

                    Log::error('Authorize.net payment failed', [
                        'order_reference' => $order->reference,
                        'error_code' => $responseCode,
                        'error_message' => $errorMessage,
                        'has_transaction_response' => $tresponse !== null
                    ]);

                    return [
                        'success' => false,
                        'error' => $errorMessage,
                        'response_code' => $responseCode
                    ];
                }
            } else {
                Log::error('Authorize.net payment failed - no response', [
                    'order_reference' => $order->reference
                ]);

                return [
                    'success' => false,
                    'error' => 'No response from payment gateway'
                ];
            }

        } catch (\Exception $e) {
            Log::error('Authorize.net payment exception', [
                'order_reference' => $order->reference,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => 'Payment processing error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get payment gateway configuration for frontend
     */
    public function getConfig(): array
    {
        return [
            'client_key' => $this->getClientKey(),
            'accept_js_url' => $this->getAcceptJsUrl(),
        ];
    }

    /**
     * Get available payment methods for this gateway
     */
    public function getPaymentMethods(): array
    {
        $methods = ['credit_card'];
        
        if ($this->settings->get('authnet.pay_by_check', false)) {
            $methods[] = 'pay_by_check';
        }
        
        return $methods;
    }

    /**
     * Get merchant authentication object
     */
    private function getMerchantAuthentication(): AnetAPI\MerchantAuthenticationType
    {
        $merchantAuthentication = new AnetAPI\MerchantAuthenticationType();
        $merchantAuthentication->setName($this->apiLoginId);
        $merchantAuthentication->setTransactionKey($this->transactionKey);
        
        return $merchantAuthentication;
    }

    /**
     * Get the Client Key for Accept.js
     */
    public function getClientKey(): ?string
    {
        return $this->settings->get('authnet.client_key', '');
    }

    /**
     * Get the Accept.js URL based on environment
     */
    public function getAcceptJsUrl(): string
    {
        $baseUrl = $this->settings->get('authnet.base_url', 'https://apitest.authorize.net');
        $isSandbox = str_contains($baseUrl, 'sandbox') || str_contains($baseUrl, 'apitest');
        
        return $isSandbox 
            ? 'https://jstest.authorize.net/v1/Accept.js'
            : 'https://js.authorize.net/v1/Accept.js';
    }
}

